﻿/*
COPYRIGHT INFORMATION:
This free script was written by Daniel Rodney - instructor, graphic designer
Adobe Certified Expert
Adobe Certified Instructor
No part of this script can be used and/or redistributed as your own.
All rights are reserved by Daniel Rodney.
http://www.danrodney.com
dan@danrodney.com
Copyright Daniel Rodney 2004-2006

DESCRIPTION:
Once you create a shape, InDesign CS doesn't let you change it to another shape. For instance, 
if you create a rectangle and later want it to be a circle, there's no "easy" way to do so...
until this script. This script will change one or more objects into whatever standard InDesign 
shape you want: rectangle, oval, polygon or star. I hope you find it useful and a time saver. 

You can share this script with friends and associates as long as it does not involve posting 
it on a website or other medium where it can be downloaded/shared with many people. 
Any form of mass distribution is prohibited unless you obtain prior written consent. 
Please direct people to http://www.danrodney.com to download it.

Thanks Dan!
If you find this "free" script valuable, please consider making a donation via PayPal. It will:
- Encourage me to write more free scripts.
- Make you (and me) feel warm and fuzzy inside.
- Be a great way to say thanks for the hours I spent writing a script that saved you some time and effort.
Just visit http://www.danrodney.com/scripts
Go to the Change Object Shape page and click the Make A Donation button. Thanks!

COMPATIBILITY:
This script works on both Mac and PC.
It is compatible with InDesign CS (not CS2). Even though InDesign CS2 added a feature that does 
essentially the same function as this script, I have made a new CS2 compatible version of this script.
Please visit www.danrodney.com to download a copy.

INSTRUCTIONS - INSTALLING THE SCRIPT:
It should be copied into the Presets > Scripts folder where you installed InDesign.
That typically means...
Mac users copy it into:
    Hard Drive > Applications > Adobe InDesign CS > Presets > Scripts
PC (Windows) users copy it into:
    C:Drive > Program Files > Adobe > Adobe InDesign CS > Presets > Scripts

INSTRUCTIONS - USING THE SCRIPT:
1. Select a frame (or frames) who's shape you want to change.
2. Double-click this script in the Scripts palette
   (to open it go into the Window menu and choose Scripting > Scripts)
3. Choose the type of shape you want and click OK.
   NOTE: If you chose Polygon/Star a second dialog will open asking how many sides and how much
   Star Inset you want. The initial values are your own Polygon Preferences you set when
   double-clicking the Polygon tool.

IMPORTANT NOTICE!!!!
If you need to "undo" the script you will need to undo several things. The whole script CANNOT
be undone in one step. This is a limitation of InDesign, not my script. The script actually:
1. creates a new correctly shaped object in the top left corner of the page
2. sizes and positions it correctly
3. deletes it
4. changes the current shape to the new shape
So if you want to undo the script each of the steps must be undone.
Be careful about small shapes being leftover in the top left corner of the page if you do not undo
enough times. The easiest way to "undo" the script is probably to run the script again and change the 
shape back to the original shape. That works fine unless you changed numerous objects of various shapes.

TIPS:
1) Use the Direct Selection tool (white arrow) instead of the Selection tool (black arrow)
when selecting objects. This allows the script to work on groups of objects. It also lets you change
the shape of an object within a group.
2) When adjusting the polygon preferences (amount of sides and star inset) you can use the 
up or down arrow keys to change the values. Shift-up arrow or Shift-down arrow goes in larger increments.
(NOTE: Those are standard Adobe keystrokes that work most of the time in InDesign, Illustrator or Photoshop.)
==================================================================================================
*/

//start of the actual script
//alert(app.selection[0].parent.constructor.name)

function myChangeShapeFunction()
{
	if (app.documents.length == 0) //make sure a document is open
	{
		//no documents are open, so display an error message
		alert("You don't have any InDesign documents open. Please open a document and select an object who's shape you want to change.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;  //this stops the function from proceeding, so the user can deal with the error
	}
	else if (app.selection.length == 0) //make sure something is selected
	{
		//nothing is selected, so display an error message
		alert("You have nothing selected. Please select an object who's shape you want to change.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
	}
	
	for (i=0; i < app.selection.length; i++) //loop over each object that is selected, making sure nothing bad is selected
	{
		if (app.selection[i].constructor.name == "Group") //make sure a group is not selected
		{
			alert("You have a group selected. Please do either of the following: A) Use the Direct Selection tool to re-select the objects or B) Ungroup them. Then run this script again.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
		else if (app.selection[i].parent.constructor.name == "Character") //make sure the frame is not anchored
		{
			alert("You have an anchored frame selected. Please unanchor the frame before running this script.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
		else if ( app.selection[i].parent.constructor.name == "Polygon")  //make sure the object isn't nested into something else (something I might have thought of)
		{
			alert("You have something other than a regular object selected. Please select a frame (or frames) before running this script.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
		else if ( app.selection[i].parent.constructor.name == "Oval")  //make sure the object isn't nested into something else (something I might have thought of)
		{
			alert("You have something other than a regular object selected. Please select a frame (or frames) before running this script.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
		else if ( app.selection[i].parent.constructor.name == "Rectangle")  //make sure the object isn't nested into something else (something I might have thought of)
		{
			alert("You have something other than a regular object selected. Please select a frame (or frames) before running this script.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
		else if ( app.selection[i].parent.constructor.name == "Story")  //make sure the object isn't nested into something else (something I might have thought of)
		{
			alert("You have text selected. Please select a frame (or frames) before running this script.\nScript written by Daniel Rodney - instructor, graphic designer\nwww.danrodney.com");
			return false;
		}
	}

	//Save the user's original polygon preferences, so I can user them in the script as well as reset them at the end of the script
	origUserPolygonPreferenceNumberOfSides = app.activeDocument.polygonPreferences.numberOfSides
	origUserPolygonPreferenceInset = app.activeDocument.polygonPreferences.insetPercentage
	
	//Create the dialog box. (The instruction to display this dialog will come later.)
	myDialog = app.dialogs.add( {name:"Change Object Shape — by www.danrodney.com", canCancel:true} );
	with( myDialog )
	{
		with( dialogColumns.add() )  //Add a dialog column
		{
			with( borderPanels.add() )  //Create a border panel
			{
				with( dialogColumns.add() )
				{
					staticTexts.add( {staticLabel:"Change Object Shape To:"} );  //Create a text label
				}
				with( dialogColumns.add() ) 
				{
					with( myRadioButtonGroup = radiobuttonGroups.add() ) //Create a group of radio buttons
					{
						myRadioButtonRectangle = radiobuttonControls.add( {staticLabel:"Rectangle", checkedState:true } );
						myRadioButtonOval = radiobuttonControls.add( {staticLabel:"Oval"} );
						myRadioButtonPolygon = radiobuttonControls.add( {staticLabel:"Polygon/Star"} );
					}
				}
			}
		}
	}
		
	//Display the dialog box.
	if( myDialog.show() == true) //If the user does NOT cancel, then change the shape. Otherwise do nothing.
	{
		//Test to see which radio button is checked and set the desired object based on that.
		if( myRadioButtonGroup.selectedButton == 0 )
		{
			myDesiredShape = "Rectangle";
		}
		else if( myRadioButtonGroup.selectedButton == 1 )
		{
			myDesiredShape = "Oval";
		}
		else
		{
			myPolygonDialog = app.dialogs.add( {name:"Polygon Preferences — by www.danrodney.com", canCancel:false} );
			with( myPolygonDialog )
			{
				with( dialogColumns.add() )
				{
					with( borderPanels.add() )
					{
						with( dialogColumns.add() ) 
						{
							staticTexts.add( {staticLabel:"Number of Sides on Polygon:"} );
							staticTexts.add( {staticLabel:"Star Inset:"} );
						}
						with( dialogColumns.add() ) 
						{
							myPolygonNumberOfSidesField = integerEditboxes.add( {editValue:origUserPolygonPreferenceNumberOfSides, minWidth:68, smallNudge:1, largeNudge:5, minimumValue:3, maximumValue:100} )
							myDesiredPolygonInsetField = percentEditboxes.add( {editValue:origUserPolygonPreferenceInset, minWidth:68, smallNudge:1, largeNudge:10, minimumValue:0, maximumValue:100} )
						}
					}
				}
			}

			myPolygonDialog.show()   //show the polygon preferences dialog box
			
			myDesiredShape = "Polygon";
			myDesiredPolygonNumberOfSides = myPolygonNumberOfSidesField.editValue;
			myDesiredPolygonInset = myDesiredPolygonInsetField.editValue;

		}
	
		for (i=0; i < app.selection.length; i++)	//loop over all the currently selected objects
		{
			mySelectedObject = app.selection[i];
		
			//get size & position of the selected object
			myBounds = mySelectedObject.geometricBounds;   
		
			//create a temporary object to that is like the one I want
			if (myDesiredShape == "Rectangle")
			{
				myTemporaryObject = mySelectedObject.parent.rectangles.add();
			}
			else if (myDesiredShape == "Oval")
			{
				myTemporaryObject = mySelectedObject.parent.ovals.add();
			}
			else if (myDesiredShape == "Polygon")
			{	
				//set the polygon preferences so when I create the polygon it will be created properly
				app.activeDocument.polygonPreferences.numberOfSides = myDesiredPolygonNumberOfSides;
				app.activeDocument.polygonPreferences.insetPercentage = myDesiredPolygonInset;

				myTemporaryObject = mySelectedObject.parent.polygons.add();
			}
		
			//set the temporary object's attributes
			myTemporaryObject.geometricBounds = myBounds;
			
			//save the path of the temporary object for later use
			myPath = myTemporaryObject.paths[0].entirePath;
		
			//remove the temporary object
			myTemporaryObject.remove();	
		
			//change the currently selected object's path to the one I had created with the temporary path
			mySelectedObject.paths[0].entirePath = myPath;	
		}
		
		//remove the dialog from memory now that the script is done
		myDialog.destroy();
		
		//reset the user's original polygon preferences
		app.activeDocument.polygonPreferences.numberOfSides = origUserPolygonPreferenceNumberOfSides;
		app.activeDocument.polygonPreferences.insetPercentage = origUserPolygonPreferenceInset;

	}
}

//Execute the above function to run the script.
//Explanation: I made the whole script a function so I could return false to stop the function as needed.
myChangeShapeFunction();